<?php

namespace Usoko\SIGBundle\Controller;

use Symfony\Component\HttpFoundation\Request;
use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Usoko\SIGBundle\Entity\CreditoPedido;
use Usoko\SIGBundle\Form\CreditoPedidoType;
use Usoko\SIGBundle\General\AuditoriaGeneral;
use Usoko\SIGBundle\General\CryptGeneral;
use Usoko\SIGBundle\General\Validacao;
use Usoko\SIGBundle\General\VarsGeneral;
use Usoko\SIGBundle\General\DataGeneral;
use Usoko\SIGBundle\General\UploadGeneral;
use Symfony\Component\Form\FormError;

/**
 * CreditoPedido controller.
 *
 */
class CreditoPedidoController extends Controller {

    /**
     * Lists all CreditoPedido entities.
     *
     */
    public function indexAction($cod, Request $request) {

        $em = $this->getDoctrine()->getManager();

        if ($cod == 'P') {
            $estado = 'A';      // Aprovado N/Pago
        } elseif ($cod == 'E') {
            $estado = 'R';      // Reprovado
        } elseif ($cod == 'PX') {
            $estado = 'AP';     // Aprovado Pago
        } elseif ($cod == 'CG') {
            $estado = 'C';      // Aprovado Pago
        } else {
            $estado = 'P';      // Pendente
        }

        $entities = $em->getRepository('UsokoSIGBundle:CreditoPedido')->findBy(array('estado' => $estado));

        $limit_page = 10;
        $paginator = $this->get('knp_paginator');
        $pagination = $paginator->paginate(
                $entities, $request->query->getInt('page', 1), $limit_page
        );

        return $this->render('UsokoSIGBundle:CreditoPedido:index.html.twig', array(
                    'pagination' => $pagination,
                    'page' => $request->query->getInt('page', 1),
                    'limit_page' => $limit_page,
                    'estado' => $estado));
    }

    /**
     * Creates a new CreditoPedido entity.
     *
     */
    public function createAction(Request $request, $associado) {

        $associadoCR = $associado;
        $associado = CryptGeneral::decrypt($this, $associadoCR);
        $entity = new CreditoPedido();

        $form = $this->createCreateForm($entity, $associado);
        $form->handleRequest($request);

        $em = $this->getDoctrine()->getManager();

        $associado = $em->getRepository('UsokoSIGBundle:Associado')->find($associado);
        if (!$associado) {
            throw $this->createNotFoundException('Associado Não Existe');
        }

        $ano = date('Y');
        $plafound = new \Usoko\SIGBundle\Entity\CreditoPlafond();
        $plafound = $em->getRepository('UsokoSIGBundle:CreditoPlafond')->findBy(array('ano' => $ano));

        if (!isset($plafound[0])) {
            return $this->render('UsokoSIGBundle:Error:CreditoConfig.html.twig');
        }

        $valid = true;
        // ----------------------------------------------- 
        $help = $request->request->get('help');
        $entity->setTempoAmortizacao($help);
        // -----------------------------------------------
        // 
        // Dois Creditos ao Mesmo Tempo
        $dql = "SELECT cp "
                . "FROM UsokoSIGBundle:creditoPedido cp "
                . "JOIN cp.associado a "
                . "JOIN cp.tipoCredito tc "
                . "WHERE cp.estado NOT IN ('R','C') "
                . " AND a.id = :associado "
                . " AND tc.id = :tipocredito ";

        $query = $em->createQuery($dql)
                ->setParameter('associado', $associado->getId())
                ->setParameter('tipocredito', $entity->getTipoCredito()->getId());
        $pendentes = $query->getResult();

        if (count($pendentes) > 0) {
            $tipo = $entity->getTipoCredito()->getNome();
            $form->get('tipoCredito')->addError(new FormError("Associado com Crédito $tipo em Processamento"));
            $valid = false;
        }

        if ($valid) {

            $valorpagar = ($entity->getValor() + ($entity->getValor() * $plafound[0]->getTaxajuros()) / 100);
            $valorMensal = $valorpagar / $entity->getTempoAmortizacao();

            $entity->setValorMensal($valorMensal);
            $entity->setValorpagar($valorpagar);

            if ($associado->getSalario() <= $entity->getValorMensal()) {
                $form->get('valorMensal')->addError(new FormError('Valor Mensal superior ao salário'));
                $valid = false;
            }

            $maximo = $entity->getTipoCredito()->getValormaximo();
            if ($maximo < $entity->getValor()) {
                $form->get('valor')->addError(new FormError('Valor máximo permitido para este tipo de Crédito ' . $maximo));
                $valid = false;
            }
        }

        if ($valid) {
            $entity->setAssociado($associado);
            $entity->setEstado(VarsGeneral::CONS_CRED_ESTADO_PENDENTE);
            $entity->setDataActual(DataGeneral::tempoGeral());
            $entity->setValor(str_replace('.', '', $entity->getValor()));
            $entity->setCodigo($this->codigoGerador());

            $temp_anexo = $entity->getAnexo();
            $entity->setAnexo(null);

            $em->getConnection()->beginTransaction();
            $em->persist($entity);
            $em->flush();

            if ($temp_anexo) {
                $file_name = md5($entity->getId());
                $answer = UploadGeneral::upload($temp_anexo, VarsGeneral::CONS_UPLOAD_CREDITO_ANEXO, $file_name, false);
                $entity->setAnexo($answer);
            }

            $em->flush();   // update 
            AuditoriaGeneral::auditar($this, $em, $entity, "CreditoPedido", "I", "Registo de Um Novo Credito Pedido");
            $em->getConnection()->commit();
            $routa = $this->generateUrl('relatorio_tabelaAmortizacaoss', array('associado' => $associadoCR));
            $routa .='?q=' . $entity->getCodigo();
            return $this->redirect($routa);
            // return $this->redirect($this->generateUrl('myCreditoPedido_AS', array('id' => $associadoCR, 'play' => '25')));
        }

        return $this->render('UsokoSIGBundle:CreditoPedido:PA_NovoCred.html.twig', array(
                    'pp' => $entity,
                    'entity' => $associado,
                    'form' => $form->createView(),
                    'taxa' => $plafound[0]->getTaxajuros(),
        ));
    }

    /**
     * Creates a form to create a CreditoPedido entity.
     *
     * @param CreditoPedido $entity The entity
     *
     * @return \Symfony\Component\Form\Form The form
     */
    private function createCreateForm(CreditoPedido $entity, $associado) {
        $form = $this->createForm(new CreditoPedidoType(), $entity, array(
            'action' => $this->generateUrl('creditopedido_create', array('associado' => $associado)),
            'method' => 'POST',
        ));

        $form->add('submit', 'submit', array('label' => 'Registar'));

        return $form;
    }

    /**
     * Displays a form to create a new CreditoPedido entity.
     *
     */
    public function newAction($id) {
        $id = $this->get('nzo_url_encryptor')->decrypt($id);
        $em = $this->getDoctrine()->getManager();

        $entity = $em->getRepository('UsokoSIGBundle:Associado')->findOneBy(array('id' => $id));

        if (!$entity) {
            throw $this->createNotFoundException('Associado Não Encontrado...');
        }

        $creditopedido = new CreditoPedido();
        $form = $this->createCreateForm($creditopedido, $id);

        $ano = date('Y');
        $plafound = new \Usoko\SIGBundle\Entity\CreditoPlafond();
        $plafound = $em->getRepository('UsokoSIGBundle:CreditoPlafond')->findBy(array('ano' => $ano));

        if (!isset($plafound[0])) {
            return $this->render('UsokoSIGBundle:Error:CreditoConfig.html.twig');
        }

        return $this->render('UsokoSIGBundle:CreditoPedido:PA_NovoCred.html.twig', array(
                    'entity' => $entity,
                    'form' => $form->createView(),
                    'taxa' => $plafound[0]->getTaxajuros(),
        ));
    }

    /**
     * Finds and displays a CreditoPedido entity.
     *
     */
    public function showAction($id) {
        $em = $this->getDoctrine()->getManager();

        $entity = $em->getRepository('UsokoSIGBundle:CreditoPedido')->find($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find CreditoPedido entity.');
        }

        $deleteForm = $this->createDeleteForm($id);

        return $this->render('UsokoSIGBundle:CreditoPedido:show.html.twig', array(
                    'entity' => $entity,
                    'delete_form' => $deleteForm->createView(),
        ));
    }

    /**
     * Displays a form to edit an existing CreditoPedido entity.
     *
     */
    public function editAction($id) {
        $em = $this->getDoctrine()->getManager();

        $entity = $em->getRepository('UsokoSIGBundle:CreditoPedido')->find($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find CreditoPedido entity.');
        }

        $editForm = $this->createEditForm($entity);
        $deleteForm = $this->createDeleteForm($id);

        return $this->render('UsokoSIGBundle:CreditoPedido:edit.html.twig', array(
                    'entity' => $entity,
                    'edit_form' => $editForm->createView(),
                    'delete_form' => $deleteForm->createView(),
        ));
    }

    /**
     * Creates a form to edit a CreditoPedido entity.
     *
     * @param CreditoPedido $entity The entity
     *
     * @return \Symfony\Component\Form\Form The form
     */
    private function createEditForm(CreditoPedido $entity) {
        $form = $this->createForm(new CreditoPedidoType(), $entity, array(
            'action' => $this->generateUrl('creditopedido_update', array('id' => $entity->getId())),
            'method' => 'PUT',
        ));

        $form->add('submit', 'submit', array('label' => 'Update'));

        return $form;
    }

    /**
     * Edits an existing CreditoPedido entity.
     *
     */
    public function updateAction(Request $request, $id) {
        $em = $this->getDoctrine()->getManager();

        $entity = $em->getRepository('UsokoSIGBundle:CreditoPedido')->find($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find CreditoPedido entity.');
        }

        $deleteForm = $this->createDeleteForm($id);
        $editForm = $this->createEditForm($entity);
        $editForm->handleRequest($request);

        if ($editForm->isValid()) {
            $em->flush();

            return $this->redirect($this->generateUrl('creditopedido_edit', array('id' => $id)));
        }

        return $this->render('UsokoSIGBundle:CreditoPedido:edit.html.twig', array(
                    'entity' => $entity,
                    'edit_form' => $editForm->createView(),
                    'delete_form' => $deleteForm->createView(),
        ));
    }

    /**
     * Deletes a CreditoPedido entity.
     *
     */
    public function deleteAction(Request $request, $id) {
        $form = $this->createDeleteForm($id);
        $form->handleRequest($request);

        if ($form->isValid()) {
            $em = $this->getDoctrine()->getManager();
            $entity = $em->getRepository('UsokoSIGBundle:CreditoPedido')->find($id);

            if (!$entity) {
                throw $this->createNotFoundException('Unable to find CreditoPedido entity.');
            }

            $em->remove($entity);
            $em->flush();
        }

        return $this->redirect($this->generateUrl('creditopedido'));
    }

    /**
     * Creates a form to delete a CreditoPedido entity by id.
     *
     * @param mixed $id The entity id
     *
     * @return \Symfony\Component\Form\Form The form
     */
    private function createDeleteForm($id) {
        return $this->createFormBuilder()
                        ->setAction($this->generateUrl('creditopedido_delete', array('id' => $id)))
                        ->setMethod('DELETE')
                        ->add('submit', 'submit', array('label' => 'Delete'))
                        ->getForm()
        ;
    }

    public function myCreditoPedidoAction($id) {

        $id = CryptGeneral::decrypt($this, $id);
        $em = $this->getDoctrine()->getManager();

        $entity = $em->getRepository('UsokoSIGBundle:Associado')->find($id);
        if (!$entity) {
            throw $this->createNotFoundException('Associado Não Encontrado .');
        }

        $entities = $em->getRepository('UsokoSIGBundle:CreditoPedido')->findByAssociado($id, array('dataActual' => 'DESC'));

        return $this->render('UsokoSIGBundle:CreditoPedido:PA_ListaCred.html.twig', array(
                    'entity' => $entity,
                    'entities' => $entities,
        ));
    }

    public function myCreditoPedidoAtAction($id) {

        $id = CryptGeneral::decrypt($this, $id);
        $em = $this->getDoctrine()->getManager();

        $entity = $em->getRepository('UsokoSIGBundle:Associado')->find($id);
        if (!$entity) {
            throw $this->createNotFoundException('Associado Não Encontrado .');
        }

        $entities = $em->getRepository('UsokoSIGBundle:CreditoPedido')->findByAssociado($id, array('dataActual' => 'DESC'));

        return $this->render('UsokoSIGBundle:CreditoPedido:Atendente/PA_ListaCred.html.twig', array(
                    'entity' => $entity,
                    'entities' => $entities,
        ));
    }

    public function showMyAction($id, $pedido) {

        $pedidoCR = $pedido;
        $id = $this->get('nzo_url_encryptor')->decrypt($id);
        $pedido = $this->get('nzo_url_encryptor')->decrypt($pedido);

        $em = $this->getDoctrine()->getManager();

        $entity = $em->getRepository('UsokoSIGBundle:Associado')->find($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find Associado entity.');
        }

        $pedido = $em->getRepository('UsokoSIGBundle:CreditoPedido')->find($pedido);

        if (!$pedido) {
            throw $this->createNotFoundException('Unable to find Solicitação entity.');
        }

        //$pedido = new CreditoPedido();
        $espera = DataGeneral::diference($pedido->getDataActual());

        $comResEntity = new \Usoko\SIGBundle\Entity\CreditoResposta();
        $cR = new CreditoRespostaController();

        $formResposta = $cR->createCreateForm($comResEntity, $pedidoCR, $this);
        $formRespostaB = $cR->createCreateFormB($comResEntity, $pedidoCR, $this);

        return $this->render('UsokoSIGBundle:CreditoPedido:PA_DadosCred.html.twig', array(
                    'entity' => $entity,
                    'solicitacao' => $pedido,
                    'formResposta' => $formResposta->createView(),
                    'formRespostaB' => $formRespostaB->createView(),
                    'espera' => $espera
        ));
    }

    // --------------------
    public function codigoGerador() {

        $mes = $this->mesSigla(DataGeneral::tempoParam("m"));
        $ano = DataGeneral::tempoParam("y");

        $data = 'C' . $mes . $ano;
        $repository = $this->getDoctrine()->getRepository('UsokoSIGBundle:CreditoPedido');

        $query = $repository->createQueryBuilder('c')
                ->where('c.codigo like :cod')
                ->setParameter('cod', "$data%")
                ->getQuery();
        $credito = $query->getResult();

        $total = count($credito) + 1;

        $rand = rand(0, 9);
        $rand2 = rand(0, 9);
        return $data . $rand . $total . $rand2;
    }

    public function mesSigla($cod) {

        $cod--;
        $sigla = ['A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L'];
        return $sigla[$cod];
    }

}
