<?php

namespace App\Http\Controllers;

use App\Http\Requests\StoreValoresRequest;
use App\Http\Requests\UpdateValoresRequest;
use App\Models\AppConfig;
use App\Models\EntidadeSys;
use App\Models\Valores;
use Exception;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\DB;


class ValoresController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        //
        $valores = Valores::orderBy('created_at', 'desc')->get();
        $entidadeVisual = EntidadeSys::getLogoByLocal(11);
        $config = AppConfig::current();
        
        return view('admin.valores.index', compact('valores', 'entidadeVisual', 'config'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
        $entidadeVisual = EntidadeSys::getLogoByLocal(11);
        $config = AppConfig::current();

        return view('admin.valores.create', compact('entidadeVisual', 'config'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreValoresRequest $request)
    {
        //
        $pathFoto = null;
        if (Gate::allows('isAdmin') || Gate::allows('isGestor') || Gate::allows('isEditor')) {
            try{
                DB::beginTransaction();

                if ($request->file('foto') != null) {
                    $foto = $request->file('foto');
                    $pathFoto = $foto->getClientOriginalName();
                    $foto->move(public_path("admin/imagens/valores"), $pathFoto);
                }                

                $valor = new Valores();
                $valor->nome = filter_var($request->nome, FILTER_SANITIZE_STRING);
                $valor->descricao = filter_var($request->descricao, FILTER_SANITIZE_STRING);
                $valor->estado = filter_var($request->estado, FILTER_SANITIZE_STRING);
                $valor->foto         = $pathFoto;
                $valor->save();

                DB::commit();

                return redirect()->route('valores.index')->with('success', 'Valor registrado com sucesso!');

            }catch(Exception $error){
                DB::rollBack();
                return redirect()->back()->withErrors('Falha ao registrar o Valor!');
            }
        }else{
            return redirect()->route('admin');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        //
        $ID = decrypt($id);
        $valor = Valores::findOrFail($ID);
        $entidadeVisual = EntidadeSys::getLogoByLocal(11);
        $config = AppConfig::current();

        return view('admin.valores.show', compact('valor', 'entidadeVisual', 'config'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        //
        $ID = decrypt($id);
        $valor = Valores::findOrFail($ID);
        $entidadeVisual = EntidadeSys::getLogoByLocal(11);
        $config = AppConfig::current();

        return view('admin.valores.edit', compact('valor', 'entidadeVisual', 'config'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateValoresRequest $request, $id)
    {
        //
        if (Gate::allows('isAdmin') || Gate::allows('isGestor') || Gate::allows('isEditor')) {
            try {
                DB::beginTransaction();

                $ID = decrypt($id);
                $valor = Valores::findOrFail($ID);
                $pathFoto = $valor->foto;

                if ($request->file('foto') != null) {
                    if ($pathFoto && file_exists(public_path("admin/imagens/valores/{$pathFoto}"))) {
                        unlink(public_path("admin/imagens/valores/{$pathFoto}"));
                    }

                    $foto = $request->file('foto');
                    $pathFoto = $foto->getClientOriginalName();
                    $foto->move(public_path("admin/imagens/valores"), $pathFoto);
                }
                
                $valor->nome = filter_var($request->nome, FILTER_SANITIZE_STRING);
                $valor->descricao = filter_var($request->descricao, FILTER_SANITIZE_STRING);
                $valor->estado = filter_var($request->estado, FILTER_SANITIZE_STRING);
                $valor->foto = $pathFoto;
                $valor->save();

                DB::commit();

                return redirect()->route('valores.index')->with('success', 'Valor atualizado com sucesso!');

            } catch (Exception $error) {
                DB::rollBack();
                return redirect()->back()->withErrors('Falha ao atualizar o Valor!');
            }
        } else {
            return redirect()->route('admin');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        //
        if (Gate::allows('isAdmin') || Gate::allows('isGestor')) {
            try {
                DB::beginTransaction();
                
                $ID = decrypt($id);
                $valor = Valores::findOrFail($ID);

                $valor->delete();
                DB::commit();

                return redirect()->route('valores.index')->with('success', 'Valor excluída com sucesso.');

            } catch (Exception $error) {
                DB::rollBack();
                return redirect()->back()->withErrors('Falha ao excluír o Valor!');
            }
        } else {
            return redirect()->route('admin');
        }
    }
}
