<?php

namespace App\Http\Controllers;

use App\Http\Requests\StoreParceirosRequest;
use App\Http\Requests\UpdateParceirosRequest;
use App\Models\AppConfig;
use App\Models\CategoriaParceiro;
use App\Models\EntidadeSys;
use App\Models\Parceiro;
use Exception;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\DB;

class ParceiroController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        //
        $parceiros = Parceiro::orderBy('created_at', 'desc')->get();
        $entidadeVisual = EntidadeSys::getLogoByLocal(11);
        $config = AppConfig::current();

        return view('admin.parceiro.index', compact('parceiros', 'entidadeVisual', 'config'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
        $categoriasParceiros = CategoriaParceiro::where('estado', 1)->get();
        $entidadeVisual = EntidadeSys::getLogoByLocal(11);
        $config = AppConfig::current();

        return view('admin.parceiro.create', compact('categoriasParceiros', 'entidadeVisual', 'config'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreParceirosRequest $request)
    {
        //
        $pathFoto = null;
        if (Gate::allows('isAdmin') || Gate::allows('isGestor') || Gate::allows('isEditor')) {
            try{
                DB::beginTransaction();

                if ($request->file('foto') != null) {
                    $foto = $request->file('foto');
                    $pathFoto = $foto->getClientOriginalName();
                    $foto->move(public_path("admin/imagens/parceiros"), $pathFoto);
                }

                $parceiro = new Parceiro();
                $parceiro->nome = filter_var($request->nome, FILTER_SANITIZE_STRING);
                $parceiro->endereco = filter_var($request->endereco, FILTER_SANITIZE_STRING);
                $parceiro->telefone = filter_var($request->telefone, FILTER_SANITIZE_STRING);
                $parceiro->site = filter_var($request->site, FILTER_SANITIZE_STRING);
                $parceiro->categoria_id = filter_var($request->categoria, FILTER_SANITIZE_STRING);
                $parceiro->estado = filter_var($request->estado, FILTER_SANITIZE_STRING);
                $parceiro->imagem         = $pathFoto;
                $parceiro->save();

                DB::commit();

                return redirect()->route('parceiro.index')->with('success', 'Parceiro registrado com sucesso!');

            }catch(Exception $error){
                DB::rollBack();
                return redirect()->back()->withErrors('Falha ao registrar o Parceiro!');
            }
        }else{
            return redirect()->route('admin');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        //
        $ID = decrypt($id);
        $parceiro = Parceiro::findOrFail($ID);
        $entidadeVisual = EntidadeSys::getLogoByLocal(11);
        $config = AppConfig::current();

        return view('admin.parceiro.show', compact('parceiro', 'entidadeVisual', 'config'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        //
        $ID = decrypt($id);
        $parceiro = Parceiro::findOrFail($ID);
        $categoriasParceiros = CategoriaParceiro::where('estado', 1)->get();
        $entidadeVisual = EntidadeSys::getLogoByLocal(11);
        $config = AppConfig::current();

        return view('admin.parceiro.edit', compact('parceiro','categoriasParceiros', 'entidadeVisual', 'config'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateParceirosRequest $request, $id)
    {
        //
        if (Gate::allows('isAdmin') || Gate::allows('isGestor') || Gate::allows('isEditor')) {
            try {
                DB::beginTransaction();
                $ID = decrypt($id);
                $parceiro = Parceiro::findOrFail($ID); 
                $pathFoto = $parceiro->imagem;

                if ($request->file('foto') != null) {
                    if ($pathFoto && file_exists(public_path("admin/imagens/parceiros/{$pathFoto}"))) {
                        unlink(public_path("admin/imagens/parceiros/{$pathFoto}"));
                    }

                    $foto = $request->file('foto');
                    $pathFoto = $foto->getClientOriginalName();
                    $foto->move(public_path("admin/imagens/parceiros"), $pathFoto);
                }

                // Atualiza os dados do usuário
                $parceiro->nome = filter_var($request->nome, FILTER_SANITIZE_STRING);
                $parceiro->endereco = filter_var($request->endereco, FILTER_SANITIZE_STRING);
                $parceiro->telefone = filter_var($request->telefone, FILTER_SANITIZE_STRING);
                $parceiro->site = filter_var($request->site, FILTER_SANITIZE_STRING);
                $parceiro->categoria_id = filter_var($request->categoria, FILTER_SANITIZE_STRING);
                $parceiro->estado = filter_var($request->estado, FILTER_SANITIZE_STRING);
                $parceiro->imagem         = $pathFoto;
                $parceiro->save();

                DB::commit();

                return redirect()->route('parceiro.index')->with('success', 'Parceiro atualizado com sucesso!');
            } catch (Exception $error) {
                DB::rollBack();
                return redirect()->back()->withErrors('Falha ao atualizar O parceiro!');
            }
        } else {
            return redirect()->route('admin');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        //
        if (Gate::allows('isAdmin') || Gate::allows('isGestor')) {
            try {
                DB::beginTransaction();
                
                $ID = decrypt($id);
                $parceiro = Parceiro::findOrFail($ID);

                $parceiro->delete();
                DB::commit();

                return redirect()->route('parceiro.index')->with('success', 'Parceiro excluído com sucesso.');

            } catch (Exception $error) {
                DB::rollBack();
                return redirect()->back()->withErrors('Falha ao excluír o Parceiro!');
            }
        } else {
            return redirect()->route('admin');
        }
    }
}
