<?php

namespace App\Http\Controllers;

use App\Http\Requests\StoreNoticiaRequest;
use App\Http\Requests\UpdateNoticiaRequest;
use App\Models\AppConfig;
use App\Models\EntidadeSys;
use App\Models\Noticia;
use Exception;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\DB;


class NoticiaController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        //
        $noticias = Noticia::orderBy('created_at', 'desc')->get();
        $entidadeVisual = EntidadeSys::getLogoByLocal(11);
        $config = AppConfig::current();

        return view('admin.noticia.index', compact('noticias', 'entidadeVisual', 'config'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
        $entidadeVisual = EntidadeSys::getLogoByLocal(11);
        $config = AppConfig::current();

        return view('admin.noticia.create', compact('entidadeVisual', 'config'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreNoticiaRequest $request)
    {
        $pathFoto = null;
        if (Gate::allows('isAdmin') || Gate::allows('isGestor') || Gate::allows('isEditor')) {
            try{
                DB::beginTransaction();

                if ($request->file('foto') != null) {
                    $foto = $request->file('foto');
                    $pathFoto = $foto->getClientOriginalName();
                    $foto->move(public_path("admin/imagens/noticias"), $pathFoto);
                }

                $noticia = new Noticia();
                $noticia->titulo = filter_var($request->titulo, FILTER_SANITIZE_STRING);
                $noticia->data_publicacao = filter_var($request->data_publicacao, FILTER_SANITIZE_STRING);
                $noticia->resumo = filter_var($request->resumo, FILTER_SANITIZE_STRING);
                $noticia->corpo_noticia = $request->input('corpo_noticia');
                $noticia->destaque = filter_var($request->destaque, FILTER_SANITIZE_STRING);
                $noticia->imagem         = $pathFoto;
                $noticia->estado         = filter_var($request->estado, FILTER_SANITIZE_STRING);
                $noticia->save();

                DB::commit();

                return redirect()->route('noticia.index')->with('success', 'Notícia registrada com sucesso!');

            }catch(Exception $error){
                DB::rollBack();
                return redirect()->back()->withErrors('Falha ao registrar Notícia!');
            }
        }else{
            return redirect()->route('admin');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        //
        $ID = decrypt($id);
        $noticia = Noticia::findOrFail($ID);
        $entidadeVisual = EntidadeSys::getLogoByLocal(11);
        $config = AppConfig::current();

        return view('admin.noticia.show', compact('noticia', 'entidadeVisual', 'config'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        //
        $ID = decrypt($id);
        $noticia = Noticia::findOrFail($ID);
        $entidadeVisual = EntidadeSys::getLogoByLocal(11);
        $config = AppConfig::current();

        return view('admin.noticia.edit', compact('noticia', 'entidadeVisual', 'config'));
    }

    /**
     * Update the specified resource in storage.
     */

    public function update(UpdateNoticiaRequest $request, $id)
    {
        if (Gate::allows('isAdmin') || Gate::allows('isGestor') || Gate::allows('isEditor')) {
            try {
                DB::beginTransaction();

                $ID = decrypt($id);
                $noticia = Noticia::findOrFail($ID);
                $pathFoto = $noticia->imagem;

                if ($request->file('foto') != null) {
                    if ($pathFoto && file_exists(public_path("admin/imagens/noticias/{$pathFoto}"))) {
                        unlink(public_path("admin/imagens/noticias/{$pathFoto}"));
                    }

                    $foto = $request->file('foto');
                    $pathFoto = $foto->getClientOriginalName();
                    $foto->move(public_path("admin/imagens/noticias"), $pathFoto);
                }

                $noticia->titulo = filter_var($request->titulo, FILTER_SANITIZE_STRING);
                $noticia->data_publicacao = filter_var($request->data_publicacao, FILTER_SANITIZE_STRING);
                $noticia->resumo = filter_var($request->resumo, FILTER_SANITIZE_STRING);
                $noticia->corpo_noticia = $request->input('corpo_noticia');
                $noticia->destaque = filter_var($request->destaque, FILTER_SANITIZE_STRING);
                $noticia->imagem = $pathFoto;
                $noticia->estado = filter_var($request->estado, FILTER_SANITIZE_STRING);
                $noticia->save();

                DB::commit();

                return redirect()->route('noticia.index')->with('success', 'Notícia atualizada com sucesso!');

            } catch (Exception $error) {
                DB::rollBack();
                return redirect()->back()->withErrors('Falha ao atualizar a Notícia!');
            }
        } else {
            return redirect()->route('admin');
        }
    }


    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        if (Gate::allows('isAdmin') || Gate::allows('isGestor')) {
            try {
                DB::beginTransaction();
                
                $ID = decrypt($id);
                $noticia = Noticia::findOrFail($ID);

                if ($noticia->imagem && file_exists(public_path("admin/imagens/noticias/{$noticia->imagem}"))) {
                    unlink(public_path("admin/imagens/noticias/{$noticia->imagem}"));
                }

                $noticia->delete();
                DB::commit();

                return redirect()->route('noticia.index')->with('success', 'Notícia excluída com sucesso.');

            } catch (Exception $error) {
                DB::rollBack();
                return redirect()->back()->withErrors('Falha ao excluír a Notícia!');
            }
        } else {
            return redirect()->route('admin');
        }
    }

}
