<?php

namespace App\Http\Controllers;

use App\Http\Requests\UpdateLegislacaoRequest;
use App\Http\Requests\StoreLegislacaoRequest;
use App\Models\AppConfig;
use App\Models\EntidadeSys;
use App\Models\Legislacao;
use Exception;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\DB;

class LegislacaoController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        //
        $legislacoes = Legislacao::orderBy('created_at', 'desc')->get();
        $entidadeVisual = EntidadeSys::getLogoByLocal(11);
        $config = AppConfig::current();

        return view('admin.Legislacao.index', compact('legislacoes', 'entidadeVisual', 'config'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
        $entidadeVisual = EntidadeSys::getLogoByLocal(11);
        $config = AppConfig::current();

        return view('admin.Legislacao.create', compact('entidadeVisual', 'config'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreLegislacaoRequest $request)
    {
        //
        $pathDoc = null;
        if (Gate::allows('isAdmin') || Gate::allows('isGestor') || Gate::allows('isEditor')) {
            try{
                DB::beginTransaction();

                if ($request->hasFile('documento')) {
                    $doc = $request->file('documento');
                    $pathDoc = $doc->getClientOriginalName();
                    $doc->move(public_path("admin/documentos/legislacao"), $pathDoc);
                    //$caminhoDocumento = $request->file('documento')->store('public/documentos');
                }

                $legislacao = new Legislacao();
                $legislacao->titulo = filter_var($request->titulo, FILTER_SANITIZE_STRING);
                $legislacao->subTitulo = filter_var($request->subtitulo, FILTER_SANITIZE_STRING);
                $legislacao->estado = filter_var($request->estado, FILTER_SANITIZE_STRING);
                $legislacao->anexo = $pathDoc;
                $legislacao->save();

                DB::commit();

                return redirect()->route('legislacoes.index')->with('success', 'Documento Legislativo registrado com sucesso!');

            }catch(Exception $error){
                DB::rollBack();
                return redirect()->back()->withErrors('Falha ao registrar o Documento Legislativo!');
            }
        }else{
            return redirect()->route('admin');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        //
        $ID = decrypt($id);
        $legislacao = Legislacao::findOrFail($ID);
        $entidadeVisual = EntidadeSys::getLogoByLocal(11);
        $config = AppConfig::current();

        return view('admin.Legislacao.show', compact('legislacao', 'entidadeVisual', 'config'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        //
        $ID = decrypt($id);
        $legislacao = Legislacao::findOrFail($ID);
        $entidadeVisual = EntidadeSys::getLogoByLocal(11);
        $config = AppConfig::current();

        return view('admin.Legislacao.edit', compact('legislacao', 'entidadeVisual', 'config'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateLegislacaoRequest $request, $id)
    {
        if (Gate::allows('isAdmin') || Gate::allows('isGestor') || Gate::allows('isEditor')) {
            try {
                DB::beginTransaction();
                $legislacao = Legislacao::findOrFail($id);
                $pathDoc = $legislacao->anexo;

                if ($request->hasFile('documento')) {
                    $doc = $request->file('documento');
                    $pathDoc = $doc->getClientOriginalName();

                    $doc->move(public_path("admin/documentos/legislacao"), $pathDoc);
                }

                $legislacao->titulo = filter_var($request->titulo, FILTER_SANITIZE_STRING);
                $legislacao->subTitulo = filter_var($request->subtitulo, FILTER_SANITIZE_STRING);
                $legislacao->estado = filter_var($request->estado, FILTER_SANITIZE_STRING);
                $legislacao->anexo = $pathDoc;

                $legislacao->save();

                DB::commit();

                return redirect()->route('legislacoes.index')->with('success', 'Documento Legislativo atualizado com sucesso!');
            } catch (Exception $error) {
                DB::rollBack();
                return redirect()->back()->withErrors('Falha ao atualizar o Documento Legislativo!');
            }
        } else {
            return redirect()->route('admin');
        }
    }


    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {        //
        if (Gate::allows('isAdmin') || Gate::allows('isGestor')) {
            try {
                DB::beginTransaction();
                
                $ID = decrypt($id);
                $legislacao = Legislacao::findOrFail($ID);
                $legislacao->delete();
                DB::commit();

                return redirect()->route('legislacoes.index')->with('success', 'Documento Legislativo excluído com sucesso.');

            } catch (Exception $error) {
                DB::rollBack();
                return redirect()->back()->withErrors('Falha ao excluír o Documento Legislativo!');
            }
        } else {
            return redirect()->route('admin');
        }
    }
}
