<?php

namespace App\Http\Controllers;

use App\Http\Requests\StoreEventosRequest;
use App\Http\Requests\UpdateEventosRequest;
use App\Models\AppConfig;
use App\Models\EntidadeSys;
use App\Models\Evento;
use App\Models\MidiaEvento;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Gate;

class EventoController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        //
        $eventos = Evento::latest()->paginate(10);
        $entidadeVisual = EntidadeSys::getLogoByLocal(11);
        $config = AppConfig::current();
        return view('admin.evento.index', compact('eventos', 'entidadeVisual', 'config'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(){
        //
        $entidadeVisual = EntidadeSys::getLogoByLocal(11);
        $config = AppConfig::current();
        return view('admin.evento.create', compact('entidadeVisual', 'config'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreEventosRequest $request)
    {
        if (Gate::allows('isAdmin') || Gate::allows('isGestor') || Gate::allows('isEditor')) {
            try{
                DB::beginTransaction();

                if ($request->file('imagem') != null) {
                    $foto = $request->file('imagem');
                    $pathFoto = $foto->getClientOriginalName();
                    $foto->move(public_path("admin/eventos"), $pathFoto);
                }

                $evento = new Evento();
                $evento->titulo = filter_var($request->titulo, FILTER_SANITIZE_STRING);
                $evento->data_evento = filter_var($request->data_evento, FILTER_SANITIZE_STRING);
                $evento->resumo = filter_var($request->resumo, FILTER_SANITIZE_STRING);
                $evento->detalhe_evento = $request->input('detalhes');
                $evento->imagem         = $pathFoto;
                $evento->estado         = filter_var($request->estado, FILTER_SANITIZE_STRING);
                $evento->save();

                DB::commit();

                return redirect()->route('eventos.index')->with('success', 'Evento criado com sucesso!');

            }catch(Exception $error){
                DB::rollBack();
                return redirect()->back()->withErrors('Falha ao registrar o Evento!');
            }
        }else{
            return redirect()->route('admin');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        $ID = decrypt($id);
        $evento = Evento::findOrFail($ID);
        $entidadeVisual = EntidadeSys::getLogoByLocal(11);
        $config = AppConfig::current();
        return view('admin.evento.show', compact('evento', 'entidadeVisual', 'config'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        //
        $ID = decrypt($id);
        $evento = Evento::findOrFail($ID);
        $entidadeVisual = EntidadeSys::getLogoByLocal(11);
        $config = AppConfig::current();
        return view('admin.evento.edit', compact('evento', 'entidadeVisual', 'config'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateEventosRequest $request, $id)
    {
        //
        if (Gate::allows('isAdmin') || Gate::allows('isGestor') || Gate::allows('isEditor')) {
            try {
                DB::beginTransaction();

                $ID = decrypt($id);
                $evento = Evento::findOrFail($ID);
                $pathFoto = $evento->imagem;

                if ($request->file('imagem') != null) {
                    if ($pathFoto && file_exists(public_path("admin/eventos/{$pathFoto}"))) {
                        unlink(public_path("admin/eventos/{$pathFoto}"));
                    }

                    $foto = $request->file('imagem');
                    $pathFoto = $foto->getClientOriginalName();
                    $foto->move(public_path("admin/eventos"), $pathFoto);
                }

                $evento->titulo = filter_var($request->titulo, FILTER_SANITIZE_STRING);
                $evento->data_evento = filter_var($request->data_evento, FILTER_SANITIZE_STRING);
                $evento->resumo = filter_var($request->resumo, FILTER_SANITIZE_STRING);
                $evento->detalhe_evento = $request->input('detalhes');
                $evento->imagem         = $pathFoto;
                $evento->estado         = filter_var($request->estado, FILTER_SANITIZE_STRING);
                $evento->save();

                DB::commit();

                return redirect()->route('eventos.index')->with('success', 'Evento atualizado com sucesso!');

            } catch (Exception $error) {
                DB::rollBack();
                return redirect()->back()->withErrors('Falha ao atualizar a Notícia!');
            }
        } else {
            return redirect()->route('admin');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        //
        if (Gate::allows('isAdmin')) {
            try {
                DB::beginTransaction();
                
                $ID = decrypt($id);
                $evento = Evento::findOrFail($ID);

                if ($evento->imagem && file_exists(public_path("admin/eventos/{$evento->imagem}"))) {
                    unlink(public_path("admin/eventos/{$evento->imagem}"));
                }

                $evento->delete();
                DB::commit();

                return redirect()->route('evento.index')->with('success', 'Evento Eliminado com sucesso.');

            } catch (Exception $error) {
                DB::rollBack();
                return redirect()->back()->withErrors('Falha ao excluír ao Evento!');
            }
        } else {
            return redirect()->route('admin');
        }
    }
}
