<?php

namespace App\Http\Controllers;

use App\Http\Requests\StoreEntidadeSysRequest;
use App\Http\Requests\UpdateEntidadeSysRequest;
use App\Models\AppConfig;
use App\Models\EntidadeSys;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Gate;

class EntidadeSysController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        //
        $entidadeDados = EntidadeSys::latest()->paginate(10);
        $entidadeVisual = EntidadeSys::getLogoByLocal(11);
        $config = AppConfig::current();

        $localList = [
            1 => "Email (Header)",
            2 => "Email (Rodapé)",
            3 => "Portal (Sobre)",
            4 => "Portal (Header)",
            5 => "Portal (Rodapé)",
            6 => "Portal (Bunner)",
            7 => "Portal (Marca de água)",
            8 => "Portal (Boas vindas - header)",
            9 => "Portal (Boas vindas - rodapé)",
            10 => "Portal (Boas vindas - Carrossel)",
            11 => "Area Reservada (Dashboard)",
            12 => "Area Reservada (Fluxo de autenticação)",                                           
            13 => "Todos",
        ];

        return view('admin.configuracoes.entidade.visual.index', compact('entidadeVisual', 'localList', 'entidadeDados', 'config'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
        $entidadeVisual = EntidadeSys::getLogoByLocal(11);
        $config = AppConfig::current();
        return view('admin.configuracoes.entidade.visual.create', compact('entidadeVisual', 'config'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreEntidadeSysRequest $request)
    {
        //
        $pathLogo = null;
        if (Gate::allows('isAdmin')) {
            try{
                DB::beginTransaction();

                $novaLocalizacao = (int) $request->local;
                $estadoAtivo = (int) $request->estado;

                if ($estadoAtivo === 1) {
                    // Verifica se já existe alguma entidade ativa com localizacao = 6 (Todos)
                    $existeTodos = EntidadeSys::where('estado', 1)->where('localizacao', 13)->first();

                    // Verifica se já existe uma entidade ativa com a mesma localizacao (exceto quando localizacao = 6)
                    $existeMesmoLocal = EntidadeSys::where('estado', 1)
                        ->where('localizacao', $novaLocalizacao)
                        ->when($novaLocalizacao === 13, function ($query) {
                            return $query; // Se for "Todos", não filtra por localização
                        }, function ($query) use ($novaLocalizacao) {
                            return $query->where('localizacao', $novaLocalizacao);
                        })
                        ->first();

                    // Regra 1: Se já existe uma entidade com local "Todos" ativa
                    if ($existeTodos && $novaLocalizacao !== 13) {
                        return redirect()->back()->withErrors('Já existe uma entidade ativa com localização "Todos". Não é possível registrar outra entidade ativa.');
                    }

                    // Regra 2: Se estou tentando registrar "Todos", mas já existe qualquer entidade ativa
                    if ($novaLocalizacao === 13) {
                        $algumaAtiva = EntidadeSys::where('estado', 1)->first();
                        if ($algumaAtiva) {
                            return redirect()->back()->withErrors('Não é possível registrar uma entidade com localização "Todos" enquanto houver outra ativa.');
                        }
                    }

                    // Regra 3: Se já existe uma entidade ativa com a mesma localização (exceto "Todos")
                    if ($existeMesmoLocal) {
                        return redirect()->back()->withErrors('Já existe uma entidade ativa com essa localização.');
                    }
                }

                if ($request->file('logo') != null) {
                    $foto = $request->file('logo');
                    $pathLogo = $foto->getClientOriginalName();
                    $foto->move(public_path("admin/endidadesIMG"), $pathLogo);
                }

                $entidade = new EntidadeSys();
                $entidade->nome = filter_var($request->nome, FILTER_SANITIZE_STRING);
                $entidade->logo = $pathLogo;
                $entidade->localizacao = filter_var($request->local, FILTER_SANITIZE_STRING);
                $entidade->estado = filter_var($request->estado, FILTER_SANITIZE_STRING);
                $entidade->save();
                DB::commit();

                return redirect()->route('entidade-visual.index')->with('success', 'Entidade registrada com sucesso!');

            }catch(Exception $error){
                DB::rollBack();
                // dd($error);
                return redirect()->back()->withErrors('Falha ao registrar a entidade');
            }
        }else{
            return redirect()->route('admin');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show($id){
        //
        $ID = decrypt($id);
        $entidade = EntidadeSys::findOrFail($ID);
        $entidadeVisual = EntidadeSys::getLogoByLocal(11);
        $config = AppConfig::current();
        return view('admin.configuracoes.entidade.visual.show',compact('entidade', 'entidadeVisual', 'config'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id){
        //
        $ID = decrypt($id);
        $entidade = EntidadeSys::findOrFail($ID);
        $entidadeVisual = EntidadeSys::getLogoByLocal(11);
        $config = AppConfig::current();

        return view('admin.configuracoes.entidade.visual.edit',compact('entidade', 'entidadeVisual', 'config'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateEntidadeSysRequest $request, $id){
        //
        $ID = decrypt($id);
        $entidade = EntidadeSys::findOrFail($ID);
        $pathLogo = $entidade->logo;

        if (Gate::allows('isAdmin')) {
            try {
                DB::beginTransaction();

                $novaLocalizacao = (int) $request->local;
                $estadoAtivo = (int) $request->estado;

                if ($estadoAtivo === 1) {
                    // Verifica se já existe uma entidade ativa com localização "Todos", excluindo a entidade atual
                    $existeTodos = EntidadeSys::where('estado', 1)
                        ->where('localizacao', 13)
                        ->where('id', '!=', $entidade->id)
                        ->first();

                    // Verifica se já existe uma entidade ativa com a mesma localização (exceto "Todos"), excluindo a atual
                    $existeMesmoLocal = EntidadeSys::where('estado', 1)
                        ->where('localizacao', $novaLocalizacao)
                        ->where('id', '!=', $entidade->id)
                        ->when($novaLocalizacao === 13, function ($query) {
                            return $query;
                        }, function ($query) use ($novaLocalizacao) {
                            return $query->where('localizacao', $novaLocalizacao);
                        })
                        ->first();

                    // Regra 1
                    if ($existeTodos && $novaLocalizacao !== 13) {
                        return redirect()->back()->withErrors('Já existe uma entidade ativa com localização "Todos". Não é possível ativar esta entidade.');
                    }

                    // Regra 2
                    if ($novaLocalizacao === 13) {
                        $algumaAtiva = EntidadeSys::where('estado', 1)
                            ->where('id', '!=', $entidade->id)
                            ->first();
                        if ($algumaAtiva) {
                            return redirect()->back()->withErrors('Não é possível ativar uma entidade com localização "Todos" enquanto houver outra ativa.');
                        }
                    }

                    // Regra 3
                    if ($existeMesmoLocal) {
                        return redirect()->back()->withErrors('Já existe uma entidade ativa com essa localização.');
                    }
                }

                if ($request->hasFile('logo')) {
                    $foto = $request->file('logo');
                    $pathLogo = $foto->getClientOriginalName();
                    $foto->move(public_path("admin/endidadesIMG"), $pathLogo);
                }

                // Atualiza os dados
                $entidade->nome = filter_var($request->nome, FILTER_SANITIZE_STRING);
                $entidade->logo = $pathLogo;
                $entidade->localizacao = filter_var($request->local, FILTER_SANITIZE_STRING);
                $entidade->estado = filter_var($request->estado, FILTER_SANITIZE_STRING);
                $entidade->save();

                DB::commit();

                return redirect()->route('entidade-visual.index')->with('success', 'Entidade atualizada com sucesso!');
            } catch (Exception $error) {
                DB::rollBack();
                // dd($error);
                return redirect()->back()->withErrors('Falha ao atualizar a entidade');
            }
        } else{
            return redirect()->route('admin');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        //
        if (Gate::allows('isAdmin')) {
            try {
                DB::beginTransaction();
                
                $ID = decrypt($id);
                $entidade = EntidadeSys::findOrFail($ID);

                if ($entidade->logo && file_exists(public_path("admin/endidadesIMG/{$entidade->logo}"))) {
                    unlink(public_path("admin/endidadesIMG/{$entidade->logo}"));
                }

                $entidade->delete();
                DB::commit();

                return redirect()->route('entidade-visual.index')->with('success', 'Entidade Visual excluída com sucesso.');

            } catch (Exception $error) {
                DB::rollBack();
                return redirect()->back()->withErrors('Falha ao excluír a Estrutura Orgânica!');
            }
        } else {
            return redirect()->route('admin');
        }
    }
}
